import re
import os
import shutil
import time

def delete_livery(aircraft_cfg_path, livery_index):
    # Open aircraft.cfg file and read its content
    with open(aircraft_cfg_path, 'r') as f:
        aircraft_cfg_content = f.read()

    # Get the name of the texture directory to delete
    texture_dir_name = get_texture_directory(aircraft_cfg_path, livery_index)

    # Remove the texture directory
    if texture_dir_name:
        texture_dir = os.path.join(os.getcwd(), texture_dir_name)
        if os.path.exists(texture_dir):
            shutil.rmtree(texture_dir)
            print(f"Removed texture directory: {texture_dir}")
        else:
            print(f"Texture directory {texture_dir} does not exist.")
    else:
        print("Texture directory name not found.")

    # Remove the livery section from the list of liveries in aircraft.cfg
    aircraft_cfg_content = re.sub(r'\[FLTSIM\.{}.*?\](.*?)(?=\[FLTSIM\.|\Z)'.format(livery_index), '', aircraft_cfg_content, flags=re.DOTALL)

    # Save the changes to the aircraft.cfg file
    with open(aircraft_cfg_path, 'w') as f:
        f.write(aircraft_cfg_content)

    # Update livery indices
    update_livery_indices(aircraft_cfg_path, livery_index)

def update_livery_indices(aircraft_cfg_path, deleted_index):
    with open(aircraft_cfg_path, 'r') as f:
        lines = f.readlines()

    # Update livery indices
    for i, line in enumerate(lines):
        match = re.match(r'\[FLTSIM\.(\d+)\]', line)
        if match:
            index = int(match.group(1))
            if index > deleted_index:
                lines[i] = re.sub(r'\[FLTSIM\.(\d+)\]', '[FLTSIM.{}]'.format(index-1), lines[i])

    # Save the changes to the aircraft.cfg file
    with open(aircraft_cfg_path, 'w') as f:
        f.write(''.join(lines))

def display_liveries(aircraft_cfg_path):
    # Clear the screen
    os.system('cls' if os.name == 'nt' else 'clear')

    # Open aircraft.cfg file and read its content
    with open(aircraft_cfg_path, 'r') as f:
        aircraft_cfg_content = f.read()

    # Display the list of liveries from the aircraft.cfg file
    pattern = r'\[FLTSIM\.(.*?)\][\s\S]*?title\s*?=\s*"(.*?)"[\s\S]*?texture\s*?=\s*"(.*?)"'
    liveries = re.findall(pattern, aircraft_cfg_content, flags=re.IGNORECASE)
    
    if liveries:
        print("Available liveries:")
        for idx, (livery_index, livery_title, texture_name) in enumerate(liveries):
            print(f"{idx}: {livery_title} - Texture directory: texture.{texture_name}")
    else:
        print("No liveries found.")

    return liveries

def display_thumbnail(livery_index):
    # Get the name of the texture directory
    texture_dir_name = get_texture_directory(aircraft_cfg_path, livery_index)
    image_path = os.path.join(os.getcwd(), texture_dir_name, "thumbnail.JPG")
    if os.path.exists(image_path):
        os.system(image_path)

def get_texture_directory(aircraft_cfg_path, livery_index):
    with open(aircraft_cfg_path, 'r') as f:
        cfg_content = f.read()
    pattern = rf'\[FLTSIM\.{livery_index}\].*?texture\s*=\s*"(.*?)"'
    match = re.search(pattern, cfg_content, flags=re.DOTALL | re.IGNORECASE)
    if match:
        texture_name = match.group(1)
        return f"texture.{texture_name}"
    else:
        return None

if __name__ == "__main__":
    # Get the path to the aircraft.cfg file
    aircraft_cfg_path = "aircraft.cfg"

    while True:
        # Display the list of liveries from the aircraft.cfg file
        liveries = display_liveries(aircraft_cfg_path)

        # Ask the user which livery to delete
        livery_index = input("Select the livery number to delete (or press 'x' to exit): ")

        if livery_index.lower() == 'x':
            break
        elif livery_index.isdigit() and 0 <= int(livery_index) < len(liveries):
            livery_index = int(livery_index)
            
            # Display thumbnail image
            display_thumbnail(int(liveries[livery_index][0]))
            
            # Ask for confirmation
            confirmation = input(f"Are you sure you want to delete livery {liveries[livery_index][1]}? (Y/N): ")

            if confirmation.lower() == 'y':
                delete_livery(aircraft_cfg_path, int(liveries[livery_index][0]))
            elif confirmation.lower() != 'n':
                print("Livery deletion cancelled.")
            else:
                print("Invalid input. Please enter 'Y' to confirm or 'N' to cancel.")
        else:
            print("Invalid input. Please enter a valid livery number or 'x' to exit.")
